from Environment.Environments.Phyre.objects import *
import json
from Environment.Environments.Phyre.rendering import system_colors

class PHYRELevel:
    def __init__(self, level=None):
        self.objects = {}
        self.bodies = {}
        self.target_object = None
        self.goal_object = None
        self.action_objects = []
        self.name = "EmptyLevel"
        self.solution = None
        # print("creating", level)

        if level is not None:
            if isinstance(level, PHYRELevel):
                self.load(level)
            elif isinstance(level, str):
                self.load_from_file(level)
            elif isinstance(level, dict):
                self.load_dict(level)
            else:
                raise Exception(f"Level {level} is not a valid type")

    def load_from_file(self, level_name, with_solution=True, level_dir="levels/"):
        """
        Load a level from a JSON file and create the appropriate Box2D bodies
        :param level_name:
        :param level_dir:
        :return:
        """

        with open(f"{level_dir}/{level_name}.json", "r") as f:
            level = json.load(f)
        self.load_dict(level, with_solution)

    def load(self, level, with_solution=True):
        """
        Load a level from a PHYRELevel object and create the appropriate Box2D bodies
        :param level:
        :return:
        """

        self.objects = level.objects
        self.target_object = level.target_object
        self.action_objects = level.action_objects
        self.goal_object = level.goal_object
        self.name = level.name
        self.solution = level.solution

    def load_dict(self, level_dict, with_solution=True):
        """
        Load a level from dictionary and create the appropriate Box2D bodies
        :param level:
        :return:
        """

        self.objects = {}
        for name, obj in level_dict["objects"].items():
            if obj["type"] == "basket":
                self.objects[name] = Basket(
                    obj["x"], obj["y"], obj["scale"], obj["angle"], obj["color"], obj["dynamic"]
                )
            elif obj["type"] == "ball":
                self.objects[name] = Ball(
                    obj["x"], obj["y"], obj["radius"], obj["color"], obj["dynamic"]
                )
            elif obj["type"] == "platform":
                self.objects[name] = Platform(
                    obj["x"],
                    obj["y"],
                    obj["length"],
                    obj["angle"],
                    obj["color"],
                    obj["dynamic"],
                )
            else:
                raise Exception(f"Object {obj} is not a valid type")
        self.target_object = level_dict["target_object"]
        self.action_objects = level_dict["action_objects"]
        self.goal_object = level_dict["goal_object"]
        self.name = level_dict["name"]
        if with_solution:
            self.solution = level_dict["solution"]
        else:
            self.solution = None

    def get_object_type(self, name):
        """
        Get the type of the object with the given name
        :param name:
        :return:
        """
        if name in self.objects:
            obj_type = self.objects[name].__class__.__name__
            obj_type = obj_type.lower()
            return obj_type
        else:
            raise Exception(f"Object {name} not found in level")

    def get_color_from_name(self, name, default="gray"):
        """
        Get the color of the object if it is in the name
        If not, set to blue by default
        If the factored state dicts pass color information, this function can be removed
        :param name:
        :return:
        """
        for color in system_colors:
            if color in name:
                return color
        if default not in system_colors:
            return "blue"
        return default


    def save(self, level_name, level_dir="levels"):
        """
        Save the current level to a JSON file
        :param level_name:
        :param level_dir:
        :return:
        """
        level = {
            "name": level_name,
            "objects": {},
            "target_object": self.target_object,
            "action_objects": self.action_objects,
        }
        for name, obj in self.objects.items():
            if isinstance(obj, Basket):
                level["objects"][name] = {
                    "x": obj.x,
                    "y": obj.y,
                    "scale": obj.scale,
                    "color": obj.color,
                    "dynamic": obj.dynamic,
                    "type": "basket",
                }
            elif isinstance(obj, Ball):
                level["objects"][name] = {
                    "x": obj.x,
                    "y": obj.y,
                    "radius": obj.radius,
                    "color": obj.color,
                    "dynamic": obj.dynamic,
                    "type": "ball",
                }
            elif isinstance(obj, Platform):
                level["objects"][name] = {
                    "x": obj.x,
                    "y": obj.y,
                    "length": obj.length,
                    "angle": obj.angle,
                    "color": obj.color,
                    "dynamic": obj.dynamic,
                    "type": "platform",
                }
            else:
                raise Exception(f"Object {obj} is not a valid type")
        if not os.path.exists(level_dir):
            os.makedirs(level_dir)
        with open(f"{level_dir}/{level_name}.json", "w") as f:
            json.dump(level, f, indent=4)

    def is_valid_level(self):
        # TODO: Check for overlapping objects
        if not self.objects:
            print("No objects found in level")
            return False
        elif not self.target_object:
            print("No target found in level")
            return False
        elif not self.action_objects:
            print("No action object found in level")
            return False
        elif not self.goal_object:
            print("No goal object found in level")
            return False
        elif self.target_object not in self.objects:
            print(f"Target object {self.target_object} not found in level")
            return False
        elif self.goal_object not in self.objects:
            print(f"Goal object {self.goal_object} not found in level")
            return False
        else:
            for action_object in self.action_objects:
                if action_object not in self.objects:
                    print(f"Action {action_object} not found in level")
                    return False

        return True

    def make_level(self, world, world_size):
        # Remove bodies already there in the world
        for body in world.bodies:
            world.DestroyBody(body)
        self.bodies = {}
        # Create walls on the edges of the screen
        self.world_size = world_size
        left_wall, right_wall, top_wall, bottom_wall = create_walls(
            world, 0.01, world_size, world_size
        )
        self.bodies["left_wall"] = left_wall
        self.bodies["right_wall"] = right_wall
        self.bodies["top_wall"] = top_wall
        self.bodies["bottom_wall"] = bottom_wall

        # Make sure action objects and target objects are dynamic
        for name in self.action_objects + [self.target_object]:
            if name in self.objects: self.objects[name].dynamic = True

        # Check for each dataclass type and create the appropriate Box2D body
        for name, obj in self.objects.items():
            if isinstance(obj, Basket):
                self.bodies[name] = create_basket(world, obj, name)
            elif isinstance(obj, Ball):
                self.bodies[name] = create_ball(world, obj, name)
            elif isinstance(obj, Platform):
                self.bodies[name] = create_platform(world, obj, name)
            else:
                raise Exception(f"Object {obj} is not a valid type")

    def set_from_factored_state(self, world, factored_state):
        """
        Create the level from a factored state
        This is a dictionary containing vectorized states for each object
        Key is the object name, value is the vectorized state
        :param factored_state:
        :return:
        """
        # Remove bodies already there in the world
        for body in world.bodies:
            world.DestroyBody(body)
        self.objects = {}

        # Create walls on the edges of the screen
        left_wall, right_wall, top_wall, bottom_wall = create_walls(
            world, 0.01, self.world_size, self.world_size
        )
        self.bodies["left_wall"] = left_wall
        self.bodies["right_wall"] = right_wall
        self.bodies["top_wall"] = top_wall
        self.bodies["bottom_wall"] = bottom_wall


        for name, state in factored_state.items():
            if name == "Done":
                continue
            elif "basket" in name:
                x = state[0]
                y = state[1]
                scale = state[2]
                dynamic = state[3]
                velocity_x = state[4]
                velocity_y = state[5]
                angular_velocity = state[6]
                color = self.get_color_from_name(name, "gray")
                self.objects[name] = Basket(x, y, scale, color, dynamic)
                self.bodies[name] = create_basket(world, self.objects[name], name)
                self.bodies[name].linearVelocity = (velocity_x, velocity_y)
                self.bodies[name].angularVelocity = angular_velocity
            elif "ball" in name:
                x = state[0]
                y = state[1]
                radius = state[2]
                dynamic = state[3]
                velocity_x = state[4]
                velocity_y = state[5]
                angular_velocity = state[6]
                color = self.get_color_from_name(name, "orange")
                self.objects[name] = Ball(
                    x, y, radius, color, dynamic
                )
                self.bodies[name] = create_ball(world, self.objects[name], name)
                self.bodies[name].linearVelocity = (velocity_x, velocity_y)
                self.bodies[name].angularVelocity = angular_velocity
            elif "platform" in name:
                x = state[0]
                y = state[1]
                length = state[2]
                angle = state[3]
                dynamic = state[4]
                velocity_x = state[5]
                velocity_y = state[6]
                angular_velocity = state[7]
                color = self.get_color_from_name(name, "black")
                self.objects[name] = Platform(
                    x, y, length, angle, color, dynamic
                )
                self.bodies[name] = create_platform(world, self.objects[name], name)
                self.bodies[name].linearVelocity = (velocity_x, velocity_y)
                self.bodies[name].angularVelocity = angular_velocity
            else:
                raise Exception(f"Object type for {name} cannot be recognized")

    def null_object(self, world, obj_name):
        """
        Null out the object with the given name
        :param world:
        :param obj_name:
        """
        # Make sure the object exists
        if obj_name not in self.objects:
            raise Exception(f"Object {obj_name} not found in level")
        # Make sure not to null the basket or target
        if obj_name == "basket":
            raise Exception(f"Cannot null the basket")
        elif obj_name == self.target:
            raise Exception(f"Cannot null the target object")
        elif obj_name in self.action_objects:
            raise Exception(f"Cannot null an action object")
        else:
            # Remove the object from the world
            for body in world.bodies:
                if body.userData == obj_name:
                    world.DestroyBody(body)
            # Remove the object from the level
            del self.objects[obj_name]

    def add_object(self, world, obj, name, is_action_object=False):
        """
        Add an object to the level
        :param world:
        :param obj:
        :param name:
        :param is_action_object:
        :return:
        """

        # Make sure the object doesn't already exist
        if obj.name in self.objects:
            raise Exception(f"Object {obj.name} already exists in level")
        # Make sure the object is not the basket or target
        if obj.name == "basket":
            raise Exception(f"Cannot add the basket")
        elif obj.name == self.target:
            raise Exception(f"Cannot add the target object")
        else:
            # Add the object to the world
            if isinstance(obj, Basket):
                create_basket(world, obj, name)
            elif isinstance(obj, Ball):
                create_ball(world, obj, name)
            elif isinstance(obj, Platform):
                create_platform(world, obj, name)
            else:
                raise Exception(f"Object {obj} is not a valid type")
            # Add the object to the level
            self.objects[name] = obj
            if is_action_object:
                self.action_objects.append(name)
